<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreCargoRequest;
use App\Http\Requests\UpdateCargoRequest;
use App\Models\AppConfig;
use App\Models\Cargo;
use App\Models\EntidadeSys;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\DB;

class CargoController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        //
        $cargos = Cargo::orderBy('created_at', 'desc')->get();
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();
        return view('admin.configuracoes.cargo.index', compact('cargos', 'entidadeVisual', 'config'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();
        return view('admin.configuracoes.cargo.create', compact('entidadeVisual', 'config'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreCargoRequest $request)
    {
        //
        if (Gate::allows('isAdmin') || Gate::allows('isGestor') || Gate::allows('isEditor')) {
            try{
                DB::beginTransaction();

                $cargos = Cargo::all();
                foreach ($cargos as $cargo) {
                    if($cargo->principal == 1){
                        if($request->principal == 1){
                            return redirect()->back()->withErrors('Já existe um Cargo registrado como Principal!');
                        }
                    }
                }

                $cargo = new Cargo();
                $cargo->nome = filter_var($request->nome, FILTER_SANITIZE_STRING);
                $cargo->descricao = filter_var($request->descricao, FILTER_SANITIZE_STRING);
                $cargo->principal = filter_var($request->principal, FILTER_SANITIZE_STRING);
                $cargo->estado = filter_var($request->estado, FILTER_SANITIZE_STRING);
                $cargo->save();

                DB::commit();

                return redirect()->route('cargo.index')->with('success', 'Cargo registrado com sucesso!');

            }catch(Exception $error){
                DB::rollBack();
                return redirect()->back()->withErrors('Falha ao registrar o Cargo!');
            }
        }else{
            return redirect()->route('admin');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        //
        $ID = decrypt($id);
        $cargo = Cargo::findOrFail($ID);
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();

        return view('admin.configuracoes.cargo.show', compact('cargo', 'entidadeVisual', 'config'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        //
        $ID = decrypt($id);
        $cargo = Cargo::findOrFail($ID);
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();

        return view('admin.configuracoes.cargo.edit', compact('cargo', 'entidadeVisual', 'config'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateCargoRequest $request, $id)
    {
        //
        if (Gate::allows('isAdmin') || Gate::allows('isGestor') || Gate::allows('isEditor')) {
            try {
                DB::beginTransaction();
                
                $ID = decrypt($id);

                $cargos = Cargo::all();
                foreach ($cargos as $cargo) {
                    if($cargo->id != $ID && $cargo->principal == 1){
                        if($request->principal == 1){
                            return redirect()->back()->withErrors('Já existe um Cargo registrado como Principal!');
                        }
                    }
                }

                $cargo = Cargo::findOrFail($ID); 

                $cargo->nome = filter_var($request->nome, FILTER_SANITIZE_STRING);
                $cargo->descricao = filter_var($request->descricao, FILTER_SANITIZE_STRING);
                $cargo->estado = filter_var($request->estado, FILTER_SANITIZE_STRING);
                $cargo->principal = filter_var($request->principal, FILTER_SANITIZE_STRING);
                $cargo->save();

                DB::commit();

                return redirect()->route('cargo.index')->with('success', 'Cargo atualizado com sucesso!');
            } catch (Exception $error) {
                DB::rollBack();
                return redirect()->back()->withErrors('Falha ao atualizar o Cargo!');
            }
        } else {
            return redirect()->route('admin');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        //
        if (Gate::allows('isAdmin') || Gate::allows('isGestor')) {
            try {
                DB::beginTransaction();
                
                $ID = decrypt($id);
                $cargo = Cargo::findOrFail($ID);

                $cargo->delete();
                DB::commit();

                return redirect()->route('cargo.index')->with('success', 'Cargo excluído com sucesso.');

            } catch (Exception $error) {
                DB::rollBack();
                return redirect()->back()->withErrors('Falha ao excluír o Cargo!');
            }
        } else {
            return redirect()->route('admin');
        }
    }
}
