<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreEstruturaOrganicaRequest;
use App\Http\Requests\UpdateEstruturaOrganicaRequest;
use App\Models\AppConfig;
use App\Models\EntidadeSys;
use App\Models\EstruturaOrganica;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;


class EstruturaOrganicaController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        //
        $estruturas = EstruturaOrganica::orderBy('created_at', 'desc')->get();
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();

        return view('admin.estruturaOrganica.index', compact('estruturas', 'entidadeVisual', 'config'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();

        return view('admin.estruturaOrganica.create', compact('entidadeVisual', 'config'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreEstruturaOrganicaRequest $request)
    {
        //
        $pathFoto = null;
        if (Gate::allows('isAdmin') || Gate::allows('isGestor') || Gate::allows('isEditor')) {
            try{
                DB::beginTransaction();

                $estruturas = EstruturaOrganica::all();
                foreach ($estruturas as $estrutura) {
                    if($estrutura->estado == 1 && $request->estado == 1){
                        if(!($estrutura->id == $request->id)){
                            return redirect()->back()->withErrors('Já existe uma Estrutura Orgânica com o estado Activo!');    
                        }
                    }
                }

                if ($request->file('foto') != null) {
                    $foto = $request->file('foto');
                    $pathFoto = $foto->getClientOriginalName();
                    $foto->move(public_path("admin/imagens/estruturaOrganica"), $pathFoto);
                }

                $estrutura = new EstruturaOrganica();
                $estrutura->titulo = filter_var($request->titulo, FILTER_SANITIZE_STRING);
                $estrutura->descricao = filter_var($request->descricao, FILTER_SANITIZE_STRING);
                $estrutura->img_organigrama         = $pathFoto;
                $estrutura->estado         = filter_var($request->estado, FILTER_SANITIZE_STRING);
                $estrutura->save();

                DB::commit();

                return redirect()->route('estrutura-organica.index')->with('success', 'Estrutura Orgânica registrada com sucesso!');

            }catch(Exception $error){
                dd($error);
                DB::rollBack();
                return redirect()->back()->withErrors('Falha ao registrar Estrutura Orgânica!');
            }
        }else{
            return redirect()->route('admin');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        //
        $ID = decrypt($id);
        $estrutura = EstruturaOrganica::findOrFail($ID);
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();

        return view('admin.estruturaOrganica.show', compact('estrutura', 'entidadeVisual', 'config'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        //
        $ID = decrypt($id);
        $estrutura = EstruturaOrganica::findOrFail($ID);
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();

        return view('admin.estruturaOrganica.edit', compact('estrutura', 'entidadeVisual', 'config'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateEstruturaOrganicaRequest $request, $id)
    {
        //
        if (Gate::allows('isAdmin') || Gate::allows('isGestor') || Gate::allows('isEditor')) {
            try {
                DB::beginTransaction();

                $estruturas = EstruturaOrganica::all();
                foreach ($estruturas as $estrutura) {
                    if($estrutura->estado == 1 && $request->estado == 1){
                        return redirect()->back()->withErrors('Já existe uma Estrutura Orgânica com o estado Activo!');    
                    }
                }

                $ID = decrypt($id);
                $estrutura = EstruturaOrganica::findOrFail($ID);
                $pathFoto = $estrutura->img_organigrama;

                if ($request->file('foto') != null) {
                    if ($pathFoto && file_exists(public_path("admin/imagens/estruturaOrganica/{$pathFoto}"))) {
                        unlink(public_path("admin/imagens/estruturaOrganica/{$pathFoto}"));
                    }

                    $foto = $request->file('foto');
                    $pathFoto = $foto->getClientOriginalName();
                    $foto->move(public_path("admin/imagens/estruturaOrganica"), $pathFoto);
                }

                
                $estrutura->titulo = filter_var($request->titulo, FILTER_SANITIZE_STRING);
                $estrutura->descricao = filter_var($request->descricao, FILTER_SANITIZE_STRING);
                $estrutura->img_organigrama         = $pathFoto;
                $estrutura->estado         = filter_var($request->estado, FILTER_SANITIZE_STRING);
                $estrutura->save();

                DB::commit();

                return redirect()->route('estrutura-organica.index')->with('success', 'Estrutura Orgânica atualizada com sucesso!');

            } catch (Exception $error) {
                DB::rollBack();
                return redirect()->back()->withErrors('Falha ao atualizar a Estrutura Orgânica!');
            }
        } else {
            return redirect()->route('admin');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        //
        if (Gate::allows('isAdmin') || Gate::allows('isGestor')) {
            try {
                DB::beginTransaction();
                
                $ID = decrypt($id);
                $estrutura = EstruturaOrganica::findOrFail($ID);

                if ($estrutura->img_organigrama && file_exists(public_path("admin/imagens/estruturaOrganica/{$estrutura->img_organigrama}"))) {
                    unlink(public_path("admin/imagens/estruturaOrganica/{$estrutura->img_organigrama}"));
                }

                $estrutura->delete();
                DB::commit();

                return redirect()->route('estrutura-organica.index')->with('success', 'Estrutura Orgânica excluída com sucesso.');

            } catch (Exception $error) {
                DB::rollBack();
                return redirect()->back()->withErrors('Falha ao excluír a Estrutura Orgânica!');
            }
        } else {
            return redirect()->route('admin');
        }
    }
}
