<?php

namespace App\Http\Controllers;

use App\General\DataGeneral;
use App\Http\Requests\LoginRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Mail\ResetPassword;
use App\Mail\ResetNotification;
use App\Services\EmailLogs;
use App\Data\EmailLogData;
use App\Mail\ResetCode;
use App\Models\AppConfig;
use App\Models\EntidadeSys;
use App\Models\NotificacaoConfig;
use App\Models\Sms;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Hash;
use App\Models\User;
use App\Services\TelcoSmsService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Session;

class SecurityController extends Controller{

    protected TelcoSmsService $smsTelcoService;

    public function __construct(TelcoSmsService $smsTelcoService)
    {
        $this->smsTelcoService = $smsTelcoService;
    }
    //
    protected function login () {

        $entidadeVisual = EntidadeSys::getLogoByLocal(12);
        $entidadeBackground = EntidadeSys::getLogoByLocal(10);
        $config = AppConfig::current();
        return view('admin.security.login', compact('entidadeVisual', 'entidadeBackground', 'config'));
    }

    public function authLogin(LoginRequest $request)
    {
        $login = $request->input('username');

        $credencias = [
            filter_var($login, FILTER_VALIDATE_EMAIL) ? 'email' : 'username' => $login,
            'password' => $request->password,
            'isActive' => 1,
        ];

        $users = User::where('username', $login)->orWhere('email', $login)->first();

        if (Auth::attempt($credencias)) {
            $request->session()->regenerate();
            Auth::logoutOtherDevices($request->password);
            return redirect()->route($users->getRedirectRoute())
                ->withSuccess('Acesso Efectuado Com Sucesso!');
        }

        return redirect()->back()->withErrors(['username' => 'Nome de Utilizador ou Senha Incorretos.']);
    }

    public function authLogout(Request $request)
    {
        if (Auth::check() === true) {
            Auth::logout();
            Session::flush();
            $request->session()->invalidate();
            $request->session()->regenerateToken();
        }
        return redirect('/login');
    }

    protected function enviarLink () {

        $entidadeVisual = EntidadeSys::getLogoByLocal(12);
        $entidadeBackground = EntidadeSys::getLogoByLocal(10);
        $config = AppConfig::current();
        $notify = NotificacaoConfig::current();
        return view('admin.security.enviarLink', compact('entidadeVisual', 'entidadeBackground', 'config', 'notify'));

    }

    protected function enviarLinkStore (Request $request) {

        //dd($request->email_);

        $notify = NotificacaoConfig::current();

        $user = null;
        if($notify->tipo == '1'){
            $user = User::where('telefone', $request->username)->first();
        }else{
            $user = User::where('email', $request->username)->first();
        }

        if (!$user) {
            session()->flash('error', 'Campo vazio ou utilizador não encontrado!');
            return redirect()->back();
        }

        $user->codigoReset = DataGeneral::gerarCodigoDeAutenticacaoParaAlterarSenha();
        $user->codigoResetExpires = DataGeneral::gerarDataExpiracaoDoCodigoRecuperacaoSenha(120);
        $user->save();

        $notify = NotificacaoConfig::current();

        if (in_array($notify->tipo, [1, 3])) {           
            $data = [
                'nome' => $user->nome_completo,
                'username' => $user->username,
                'codigo' => $user->codigoReset,
            ];
            
            $telefone = $user->telefone;
            $sms = Sms::gerarMensagemCodigoRedefinicaoSMS($data);

            $this->smsTelcoService->enviarMensagem($telefone, $sms);
        }

        // Envio de E-mail
        if (in_array($notify->tipo, [2, 3])) {
            $dados = [
                'nome' => $user->name_completo,
                'username' => $user->username,
                'codigo' => $user->codigoReset,
                'assunto' => 'Alteração da Palavra Passe',
                'email' => $user->email,
            ];

            try {

                Mail::to($user->email)->send(new ResetPassword($dados));
                $logData = [
                    'email_receptor' => $user->email,
                    'email_emissor' => config('mail.from.address'),
                    'assunto' => 'Alteração da Palavra-Passe',
                    'body' =>  json_encode($dados, JSON_UNESCAPED_UNICODE),
                    'mail_class' => ResetPassword::class,
                    'mensagem_erro' => '',
                    'estado' => 1,
                ];

                EmailLogs::log(new EmailLogData($logData));

            } catch (\Throwable $th) {
                //throw $th;
                $logData = [
                    'email_receptor' => $user->email,
                    'email_emissor' => config('mail.from.address'),
                    'assunto' => 'Alteração da Palavra-Passe',
                    'body' =>  json_encode($dados, JSON_UNESCAPED_UNICODE),
                    'mail_class' => ResetPassword::class,
                    'mensagem_erro' => $th->getMessage(),
                    'estado' => 0,
                ];

                EmailLogs::log(new EmailLogData($logData));
            }
        }

        if($notify->tipo == '1'){
            return redirect()->back()->with([
                'success' => "O código de redefinição foi enviado com sucesso ao número {$user->telefone}. Ao clicar em fechar, será redirecionado à página de validação do código!",
                'username' => $user->telefone
            ]);
        }else{
            return redirect()->back()->with([
                'success', "O código de redefinição foi enviado com sucesso ao e-mail {$user->email}. Ao clicar em fechar, será redirecionado à página de validação do código!",
                'username' => $user->email
            ]);
        }

    }

    public function resetMyPalavraPasseStore(Request $request)
    {
        DB::beginTransaction();

        $notify = NotificacaoConfig::current();

        try {
            $user = User::where('email', $request->email)->first();

            $user = null;
            if($notify->tipo == '1'){
                $user = User::where('telefone', $request->username)->first();
            }else{
                $user = User::where('email', $request->username)->first();
            }

            if (!$user) {
                session()->flash('error', 'Usuário não encontrado.');
                return redirect()->back();
            }

            if (!Hash::check($request->passAnt, $user->password)) {
                session()->flash('error', 'A Palavra Passe actual que foi informada não existe.');
                return redirect()->back();
            }

            if ( $request->pass !== $request->novaPass ) {
                session()->flash('error', 'A Palavra Passe e Confirmar Palavra Passe não podem ser diferentes.');
                return redirect()->back();
            }

            //dd($request);
            if ($request->passAnt === $request->pass || $request->passAnt === $request->novaPass) {
                session()->flash('error', 'A Palavra Passe Nova não pode ser igual a actual.');
                return redirect()->back();
            }

            $user->password = Hash::make($request->novaPass);
            $user->codigoReset = DataGeneral::gerarCodigoDeAutenticacaoParaAlterarSenha();
            $user->save();
            DB::commit();

            // Envio de SMS
            if (in_array($notify->tipo, [1, 3])) {
                $data = [
                    'nomeUser' => $user->nome_completo,
                    'email' => $user->email,
                    'dataAlteracao' => DataGeneral::dataHoraActual(),
                ];

                $telefone = $user->telefone;
                $sms = Sms::gerarMensagemRedefinicaoSenhaSMS($data);

                $this->smsTelcoService->enviarMensagem($telefone, $sms);
            }

            // Envio de E-mail
            if (in_array($notify->tipo, [2, 3])) {
                $dados = [
                    'nomeUser' => $user->name_completo,
                    'email' => $user->email,
                    'dataAlteracao' => DataGeneral::dataHoraActual(),
                    'assunto' => 'Aviso de alteração de palavra passe',
                ];

                try {
                    Mail::to($user->email)->send(new ResetNotification($dados));

                    $logData = [
                        'email_receptor' => $user->email,
                        'email_emissor' => config('mail.from.address'),
                        'assunto' => 'Alteração da Palavra-Passe',
                        'body' => json_encode($dados, JSON_UNESCAPED_UNICODE),
                        'mail_class' => ResetNotification::class,
                        'mensagem_erro' => '',
                        'estado' => 1,
                    ];

                    EmailLogs::log(new EmailLogData($logData));

                } catch (\Throwable $th) {
                    $logData = [
                        'email_receptor' => $user->email,
                        'email_emissor' => config('mail.from.address'),
                        'assunto' => 'Alteração da Palavra-Passe',
                        'body' => json_encode($dados, JSON_UNESCAPED_UNICODE),
                        'mail_class' => ResetNotification::class,
                        'mensagem_erro' => $th->getMessage(),
                        'estado' => 0,
                    ];

                    EmailLogs::log(new EmailLogData($logData));
                }
            }

            session()->flash('success', 'A sua Palavra-Passe foi alterada com sucesso.');
            return redirect()->back();
        } catch (\Exception $e) {
            DB::rollBack();
            session()->flash('error', 'Ocorreu um erro ao alterar a sua Palavra-Passe. Por favor, tente novamente.');
            return redirect()->back();
        }
    }

    protected function confirmarCodigo ($username) {
        $username = decrypt($username);
        $entidadeVisual = EntidadeSys::getLogoByLocal(12);
        $entidadeBackground = EntidadeSys::getLogoByLocal(10);
        $config = AppConfig::current();
        $notify = NotificacaoConfig::current();
        return view('admin.security.confirmarCodigo', compact('username', 'entidadeVisual', 'entidadeBackground', 'config', 'notify'));
    }

    protected function confirmarCodigoStore (Request $request) {
        $codigo = $request->digit1.$request->digit2.$request->digit3.$request->digit4.$request->digit5.$request->digit6;
        $codigo = intval($codigo);
        //$user = User::where('codigoReset', $codigo)->first();
        //$user = User::where('codigoReset', $codigo)->where('email', $request->email_)->first();
        // dd($request);
        $notify = NotificacaoConfig::current(); 
        $user = null;
        if($notify->tipo == '1'){
            $user = User::getUserByTelefone($codigo, $request->username);
        }else{
            $user = User::getUserByEmail($codigo, $request->username);
        }

        // $user = User::getUser($codigo, $request->email);

        if (empty($user) || $user == null) {
            session()->flash('error', 'Utilizador não encontrado. Por favor, tente novamente!');
            return redirect()->back();
        }

        if (isset($user[0]) && ($user[0]->codigoResetExpires < DataGeneral::dataHoraActual())) {
            session()->flash('error', 'O código de redefinição expirou. Solicite um novo código.');
            return redirect()->back();
        }

        if($notify->tipo == '1'){
            return redirect()->route('auth.reset.resetars', ['username' => encrypt($user[0]->telefone)]);
        }else{
            return redirect()->route('auth.reset.resetars', ['username' => encrypt($user[0]->email)]);
        }

        // return redirect()->route('auth.reset.resetars', ['email' => encrypt($user[0]->email)]);

    }

    protected function resetPalavraPasse ($username) {

        $username = decrypt($username);
        $entidadeVisual = EntidadeSys::getLogoByLocal(12);
        $entidadeBackground = EntidadeSys::getLogoByLocal(10);
        $config = AppConfig::current();
        $notify = NotificacaoConfig::current();
        return view('admin.security.resetPassWord', compact('username', 'entidadeVisual', 'entidadeBackground', 'config', 'notify'));

    }

    protected function resetPalavraPasseStore (Request $request) {
       // dd($request);
        DB::beginTransaction();
        $notify = NotificacaoConfig::current();
        $user = null;
        if($notify->tipo == '1'){
            $user = User::where('telefone', $request->username)->first();
        }else{
            $user = User::where('email', $request->username)->first();
        }

        if (!$user) {
            session()->flash('error', 'Campo vazio ou utilizador não encontrado!');
            return redirect()->back();
        }


        $user->password = Hash::make($request->pass);
        $user->codigoReset = DataGeneral::gerarCodigoDeAutenticacaoParaAlterarSenha();
        $user->save();
        DB::commit();        

        if (in_array($notify->tipo, [1, 3])) {
            $data = [
                'nomeUser' => $user->nome_completo,
                'username' => $user->username,
                'dataAlteracao' => DataGeneral::dataHoraActual(),
            ];

            $telefone = $user->telefone;
            $sms = Sms::gerarMensagemRedefinicaoSenhaSMS($data);

            $this->smsTelcoService->enviarMensagem($telefone, $sms);
        }

        if (in_array($notify->tipo, [2, 3])) {
            $dados = [
                'nomeUser' => $user->name_completo,
                'email' => $user->email,
                'dataAlteracao' => DataGeneral::dataHoraActual(),
                'assunto' => 'Aviso de alteração de palavra passe',
            ];

            try {
                Mail::to($user->email)->send(new ResetNotification($dados));

                $logData = [
                    'email_receptor' => $user->email,
                    'email_emissor' => config('mail.from.address'),
                    'assunto' => 'Alteração da Palavra-Passe',
                    'body' =>  json_encode($dados, JSON_UNESCAPED_UNICODE),
                    'mail_class' => ResetNotification::class,
                    'mensagem_erro' => '',
                    'estado' => 1,
                ];

                EmailLogs::log(new EmailLogData($logData));

            } catch (\Throwable $th) {
                $logData = [
                    'email_receptor' => $user->email,
                    'email_emissor' => config('mail.from.address'),
                    'assunto' => 'Alteração da Palavra-Passe',
                    'body' =>  json_encode($dados, JSON_UNESCAPED_UNICODE),
                    'mail_class' => ResetNotification::class,
                    'mensagem_erro' => $th->getMessage(),
                    'estado' => 0,
                ];

                EmailLogs::log(new EmailLogData($logData));
            }
        }

        session()->flash('success', 'Palavra passe alterada com sucesso!');
        return redirect()->route('login');
    }

    protected function resetPalavraPassePerfilStore (Request $request) {
        // dd($request);
        DB::beginTransaction();
        $notify = NotificacaoConfig::current();
        $user = null;
        if($notify->tipo == '1'){
            $user = User::where('telefone', $request->username)->first();
        }else{
            $user = User::where('email', $request->username)->first();
        }

        if (!$user) {
            session()->flash('error', 'Campo vazio ou utilizador não encontrado!');
            return redirect()->back();
        }
        
        $user->password = Hash::make($request->pass);
        $user->codigoReset = DataGeneral::gerarCodigoDeAutenticacaoParaAlterarSenha();
        $user->save();
        DB::commit();       

        if (in_array($notify->tipo, [1, 3])) {
            $data = [
                'nomeUser' => $user->nome_completo,
                'email' => $user->email,
                'dataAlteracao' => DataGeneral::dataHoraActual(),
            ];

            $telefone = $user->telefone;
            $sms = Sms::gerarMensagemRedefinicaoSenhaSMS($data);

            $this->smsTelcoService->enviarMensagem($telefone, $sms);
        }
        
        // Envio de E-mail
        if (in_array($notify->tipo, [2, 3])) {
            $dados = [
                'nomeUser' => $user->name_completo,
                'email' => $user->email,
                'dataAlteracao' => DataGeneral::dataHoraActual(),
                'assunto' => 'Aviso de alteração de palavra passe',
            ];

            try {
                Mail::to($user->email)->send(new ResetNotification($dados));

                $logData = [
                    'email_receptor' => $user->email,
                    'email_emissor' => config('mail.from.address'),
                    'assunto' => 'Alteração da Palavra-Passe',
                    'body' =>  json_encode($dados, JSON_UNESCAPED_UNICODE),
                    'mail_class' => ResetNotification::class,
                    'mensagem_erro' => '',
                    'estado' => 1,
                ];

                EmailLogs::log(new EmailLogData($logData));

            } catch (\Throwable $th) {
                $logData = [
                    'email_receptor' => $user->email,
                    'email_emissor' => config('mail.from.address'),
                    'assunto' => 'Alteração da Palavra-Passe',
                    'body' =>  json_encode($dados, JSON_UNESCAPED_UNICODE),
                    'mail_class' => ResetNotification::class,
                    'mensagem_erro' => $th->getMessage(),
                    'estado' => 0,
                ];

                EmailLogs::log(new EmailLogData($logData));
            }
        }

        session()->flash('success', 'Palavra passe alterada com sucesso!');
        return redirect()->back();
    }



    protected function reeviarCodigo ($username) {

        DB::beginTransaction();
        $username = decrypt($username);
        //dd($request->email_);

        $notify = NotificacaoConfig::current();
        $user = null;
        if($notify->tipo == '1'){
            $user = User::where('telefone', $username)->first();
        }else{
            $user = User::where('email', $username)->first();
        }

        if (!$user) {
            session()->flash('error', 'Campo vazio ou utilizador não encontrado!');
            return redirect()->back();
        }

        $user->codigoReset = DataGeneral::gerarCodigoDeAutenticacaoParaAlterarSenha();
        $user->codigoResetExpires = DataGeneral::gerarDataExpiracaoDoCodigoRecuperacaoSenha(120);
        $user->save();
        DB::commit();

        if (in_array($notify->tipo, [1, 3])) {
            $data = [
                'nome' => $user->nome_completo,
                'codigo' => $user->codigoReset,
                'expiracao' => $user->codigoResetExpires,
            ];

            $telefone = $user->telefone;
            $sms = Sms::gerarMensagemReenvioCodigoConfirmacaoSMS($data);

            $this->smsTelcoService->enviarMensagem($telefone, $sms);
        }

        // Envio de E-mail
        if (in_array($notify->tipo, [2, 3])) {
            $dados = [
                'nome' => $user->name_completo,
                'codigo' => $user->codigoReset,
                'expiracao' => $user->codigoResetExpires,
                'assunto' => 'Solicitação de Código de Confirmação',
                'email' => $user->email,
            ];

            try {
                Mail::to($user->email)->send(new ResetCode($dados));

                $logData = [
                    'email_receptor' => $user->email,
                    'email_emissor' => config('mail.from.address'),
                    'assunto' => 'Solicitação de Código de Confirmação',
                    'body' =>  json_encode($dados, JSON_UNESCAPED_UNICODE),
                    'mail_class' => ResetCode::class,
                    'mensagem_erro' => '',
                    'prioridade' => 3,
                    'estado' => 1,
                ];

                EmailLogs::log(new EmailLogData($logData));

            } catch (\Throwable $th) {
                $logData = [
                    'email_receptor' => $user->email,
                    'email_emissor' => config('mail.from.address'),
                    'assunto' => 'Solicitação de Código de Confirmação',
                    'body' =>  json_encode($dados, JSON_UNESCAPED_UNICODE),
                    'mail_class' => ResetCode::class,
                    'mensagem_erro' => $th->getMessage(),
                    'prioridade' => 1,
                    'estado' => 0,
                ];

                EmailLogs::log(new EmailLogData($logData));
            }
        }

        if($notify->tipo == '1'){
            session()->flash('success', "O novo código de confirmação foi enviado com sucesso, verifique a sua caixa de SMS do número {$user->telefone}!");
            return redirect()->back();
        }else{
            session()->flash('success', "O novo código de confirmação foi enviado com sucesso, verifique a sua caixa de Email: {$user->email}!");
            return redirect()->back();
        }
    }
}
