<?php

namespace App\Http\Controllers;

use App\Data\EmailLogData;
use App\General\DataGeneral;
use App\Http\Requests\StoreUserRequest;
use App\Http\Requests\UpdateUserRequest;
use App\Mail\ResetPassword;
use App\Models\AppConfig;
use App\Models\EntidadeSys;
use App\Models\NotificacaoConfig;
use App\Models\Role;
use App\Models\Sms;
use App\Models\User;
use App\Services\EmailLogs;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use App\Services\TelcoSmsService;
use Exception;

class UserController extends Controller
{
    protected TelcoSmsService $smsTelcoService;

    public function __construct(TelcoSmsService $smsTelcoService)
    {
        $this->smsTelcoService = $smsTelcoService;
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        //
        $users = User::orderBy('created_at', 'desc')->get();
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();

        return view('admin.user.index', compact('users', 'entidadeVisual', 'config'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
        $roles = Role::all();
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();

        return view('admin.user.create', compact('roles', 'entidadeVisual', 'config'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreUserRequest $request)
    {
        $pathFoto = null;
        if (Gate::allows('isAdmin') || Gate::allows('isGestor') || Gate::allows('isEditor')) {
            try{
                DB::beginTransaction();

                if ($request->file('foto') != null) {
                    $foto = $request->file('foto');
                    $pathFoto = $foto->getClientOriginalName();
                    $foto->move(public_path("admin/imagens/users"), $pathFoto);
                }

                $user = new User();
                $user->name_completo = filter_var($request->nome, FILTER_SANITIZE_STRING);
                $user->username = filter_var($request->username, FILTER_SANITIZE_STRING);
                $user->telefone = filter_var($request->telefone, FILTER_SANITIZE_STRING);
                $user->email = filter_var($request->email, FILTER_SANITIZE_STRING);
                $user->role_id = filter_var($request->perfil, FILTER_SANITIZE_STRING);
                $user->password = Hash::make('87654321');
                $user->codigoResetExpires = DataGeneral::gerarCodigoDeAutenticacaoParaAlterarSenha();
                $user->codigoReset = DataGeneral::gerarDataExpiracaoDoCodigoRecuperacaoSenha(120);
                $user->isActive = filter_var($request->estado, FILTER_SANITIZE_STRING);
                $user->foto         = $pathFoto;
                $user->save();
                DB::commit();

                $notify = NotificacaoConfig::current();

                if (in_array($notify->tipo, [1, 3])) {
                    $data = [
                        'nome' => $user->nome_completo,
                        'username' => $user->username,
                        'codigo' => $user->codigoReset,
                        'email' => $user->email,
                    ];

                    $telefone = $user->telefone;
                    $sms = Sms::gerarMensagemNovoUser($data);

                    $this->smsTelcoService->enviarMensagem($telefone, $sms);                    
                }

                // Envio de E-mail
                if (in_array($notify->tipo, [2, 3])) {
                    $dados = [
                        'nome' => $user->nome_completo,
                        'username' => $user->username,
                        'codigo' => $user->codigoReset,
                        'assunto' => 'Alteração da Palavra Passe',
                        'email' => $user->email,
                    ];

                    try {
                        Mail::to($user->email)->send(new ResetPassword($dados));
        
                        $logData = [
                            'email_receptor' => $user->email,
                            'email_emissor' => config('mail.from.address'),
                            'assunto' => 'Alteração da Palavra-Passe',
                            'body' =>  json_encode($dados, JSON_UNESCAPED_UNICODE),
                            'mail_class' => ResetPassword::class,
                            'mensagem_erro' => '',
                            'estado' => 1,
                        ];

                        EmailLogs::log(new EmailLogData($logData));
                        
                    } catch (\Throwable $th) {
                        $logData = [
                            'email_receptor' => $user->email,
                            'email_emissor' => config('mail.from.address'),
                            'assunto' => 'Alteração da Palavra-Passe',
                            'body' =>  json_encode($dados, JSON_UNESCAPED_UNICODE),
                            'mail_class' => ResetPassword::class,
                            'mensagem_erro' => $th->getMessage(),
                            'estado' => 0,
                        ];
                    
                        EmailLogs::log(new EmailLogData($logData));
                    }
                } 

                return redirect()->route('utilizadors.index')->with('success', 'Utilizador registrado com sucesso!');

            }catch(Exception $error){
                DB::rollBack();
                return redirect()->back()->withErrors('Falha ao registrar Utilizador!');
            }
        }else{
            return redirect()->route('admin');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        //
        $ID = decrypt($id);
        $utilizador = User::findOrFail($ID);
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();

        return view('admin.user.show', compact('utilizador', 'entidadeVisual', 'config'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        //
        $ID = decrypt($id);
        $roles = Role::all();
        $utilizador = User::findOrFail($ID);
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();

        return view('admin.user.edit', compact('roles', 'utilizador', 'entidadeVisual', 'config'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateUserRequest $request, $id)
    {

        if (Gate::allows('isAdmin') || Gate::allows('isGestor') || Gate::allows('isEditor')) {
            try {
                DB::beginTransaction();
                $ID = decrypt($id);
                $user = User::findOrFail($ID);
                $pathFoto = $user->foto;
                $userEstado = $user->isActive;

                
                if ($request->file('foto') != null) {
                    if ($pathFoto && file_exists(public_path("admin/imagens/users/{$pathFoto}"))) {
                        unlink(public_path("admin/imagens/users/{$pathFoto}"));
                    }

                    $foto = $request->file('foto');
                    $pathFoto = $foto->getClientOriginalName();
                    $foto->move(public_path("admin/imagens/users"), $pathFoto);
                }

                // Atualiza os dados do usuário
                $user->name_completo = filter_var($request->nome, FILTER_SANITIZE_STRING);
                $user->username = filter_var($request->username, FILTER_SANITIZE_STRING);
                $user->telefone = filter_var($request->telefone, FILTER_SANITIZE_STRING);
                $user->email = filter_var($request->email, FILTER_SANITIZE_STRING);
                $user->role_id = filter_var($request->perfil, FILTER_SANITIZE_STRING);
                $user->isActive = $request->estado != '' ? filter_var($request->estado, FILTER_SANITIZE_STRING) : $userEstado;
                $user->foto = $pathFoto;                

                $user->save();

                DB::commit();

                return redirect()->route('utilizadors.index')->with('success', 'Utilizador atualizado com sucesso!');
            } catch (Exception $error) {
                DB::rollBack();
                dd($error);
                return redirect()->back()->withErrors('Falha ao atualizar Utilizador!');
            }
        } else {
            return redirect()->route('admin');
        }
    }


    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        if (Gate::allows('isAdmin') || Gate::allows('isGestor')) {
            try {
                DB::beginTransaction();

                $ID = decrypt($id);
                $user = User::findOrFail($ID);

                if ($user->foto && file_exists(public_path("admin/imagens/users/{$user->foto}"))) {
                    unlink(public_path("admin/imagens/users/{$user->foto}"));
                }

                $user->delete();
                DB::commit();

                return redirect()->route('utilizadors.index')->with('success', 'Utilizador excluído com sucesso.');

            } catch (Exception $error) {
                DB::rollBack();
                return redirect()->back()->withErrors('Falha ao excluír o Utilizador!');
            }
        } else {
            return redirect()->route('admin');
        }
    }

    public function verPerfil($id)
    {
        try {
            DB::beginTransaction();

            $ID = decrypt($id);
            $user = User::findOrFail($ID);
            $entidadeVisual = EntidadeSys::getLogoByLocal(11);
            $config = AppConfig::current();
            $notify = NotificacaoConfig::current();

            return View('admin.user.perfil.show', compact('user', 'entidadeVisual', 'config', 'notify'));

        } catch (Exception $error) {
            DB::rollBack();
            return redirect()->back()->withErrors('Erro inesperado!');
        }
    }

    public function dashboard()
    {
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();

        return View('admin.home.admin', compact('entidadeVisual', 'config'));
    }
}
