<?php

namespace Usoko\SIGBundle\Controller;

use Symfony\Component\HttpFoundation\Request;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Usoko\SIGBundle\Entity\Pagamento;
use Usoko\SIGBundle\Form\PagamentoType;
use Usoko\SIGBundle\General\AuditoriaGeneral;
use Usoko\SIGBundle\General\VarsGeneral;
use Symfony\Component\Form\FormError;
use Usoko\SIGBundle\General\Validacao;
use Usoko\SIGBundle\General\MathGeneral;
use Usoko\SIGBundle\General\NumeroReal;

/**
 * Pagamento controller.
 *
 */
class PagamentoController extends Controller {

    /**
     * Lists all Pagamento entities.
     *
     */
    public function indexAction($tipo, $cod, Request $request) {


        if ($tipo == 'faddaedw-W2d') {
            $tipo = 'Q';
        } elseif ($tipo == 'fadfaew-W2d') {
            $tipo = 'J';
        } elseif ($tipo == 'fadzandw-W2d') {
            $tipo = 'C';
        } elseif ($tipo == 'fadeaenw-W2d') {
            $tipo = 'PP';
        } else {
            throw $this->createNotFoundException();
        }

        if ($cod == 'l') {
            $estado = 'l';
        } elseif ($cod == 'n') {
            $estado = 'n';
        } else {
            throw $this->createNotFoundException();
        }

        $em = $this->getDoctrine()->getManager();
        $entities = $em->getRepository('UsokoSIGBundle:Pagamento')->tipoPagamento($tipo, $estado);

        $limit_page = 10;
        $paginator = $this->get('knp_paginator');
        $pagination = $paginator->paginate(
                $entities, $request->query->getInt('page', 1), $limit_page
        );

        return $this->render("UsokoSIGBundle:Pagamento:index.html.twig", array(
                    'pagination' => $pagination,
                    'page' => $request->query->getInt('page', 1),
                    'limit_page' => $limit_page,
                    'cod' => $cod,
                    'tipo' => $tipo,
        ));
    }

    /**
     * Creates a new Pagamento entity.
     *
     */
    public function createAction(Request $request, $associado) {

        $entity = new Pagamento();
        $form = $this->createCreateForm($entity, $associado);
        $form->handleRequest($request);

        $em = $this->getDoctrine()->getManager();

        $associadoCR = $associado;
        $associado = $this->get('nzo_url_encryptor')->decrypt($associado);
        $associado = $em->getRepository('UsokoSIGBundle:Associado')->find($associado);
        if (!$associado) {
            throw $this->createNotFoundException('Associado Não Encontrado');
        }

        $entity->setAssociado($associado);

        $entity->setDataActual(\Usoko\SIGBundle\General\DataGeneral::tempoGeral());
        $entity->setEstado('L');
        $entity->setValor((double) $this->valor($request));

        $salario = $entity->getAssociado()->getSalario();

        $rs = $this->redirectPayment($entity, $em);

        if (!$rs['return']) {
            $form->get($rs['campo'])->addError(new FormError($rs['msg']));
        } else {
            $entity = $rs['entity'];
        }

        if (($rs['return']) && Validacao::validar($this, $entity)) {

            $em->getConnection()->beginTransaction();

            $em->persist($entity);
            $em->flush();

            if ($entity->getTipopagamento()->getTipo() == 'J' || $entity->getTipopagamento()->getTipo() == 'Q' || $entity->getTipopagamento()->getTipo() == 'PP') {
                $associado->setSaldoliquido($associado->getSaldoliquido() + $entity->getValor());
                $em->persist($entity);
                $em->flush();
            }

            if ($this->associadoTObeneficiario($associado->getId())) {
                $associado->setTipo(VarsGeneral::CONS_ASSOCIADO_TIPO_BENETITULAR);
                $em->persist($associado);
                $em->flush();
            }

            AuditoriaGeneral::auditar($this, $em, $entity, "Pagamento", "I", "Registo de Um Novo Pagamento");
            $em->getConnection()->commit();

            return $this->redirect($this->generateUrl('myPagamento_AS', array('id' => $associadoCR, 'play' => '21')));
        }

        $dados = $this->pagamentoInfo($salario);
        return $this->render('UsokoSIGBundle:Pagamento:PA_NovoPag.html.twig', array(
                    'entity' => $associado,
                    'form' => $form->createView(),
                    'dados' => $dados,
        ));
    }

    /**
     * Creates a form to create a Pagamento entity.
     *
     * @param Pagamento $entity The entity
     *
     * @return \Symfony\Component\Form\Form The form
     */
    private function createCreateForm(Pagamento $entity, $associado) {
        $form = $this->createForm(new PagamentoType(), $entity, array(
            'action' => $this->generateUrl('pagamento_create', array('associado' => $associado)),
            'method' => 'POST',
        ));

        $form->add('submit', 'submit', array('label' => 'Registar'));

        return $form;
    }

    /**
     * Displays a form to create a new Pagamento entity.
     *
     */
    public function newAction($id) {

        $idCR = $id;
        $id = $this->get('nzo_url_encryptor')->decrypt($id);

        $em = $this->getDoctrine()->getManager();
        $entity = $em->getRepository('UsokoSIGBundle:Associado')->find($id);
        if (!$entity) {
            throw $this->createNotFoundException('Associado Não Existe');
        }

        $pagamento = new Pagamento();
        $form = $this->createCreateForm($pagamento, $idCR);

        $dados = $this->pagamentoInfo($entity->getSalario());
        return $this->render('UsokoSIGBundle:Pagamento:PA_NovoPag.html.twig', array(
                    'entity' => $entity,
                    'form' => $form->createView(),
                    'dados' => $dados,
        ));
    }

    public function new2Action($id) {

        $idCR = $id;
        $id = $this->get('nzo_url_encryptor')->decrypt($id);

        $em = $this->getDoctrine()->getManager();
        $entity = $em->getRepository('UsokoSIGBundle:Associado')->find($id);
        if (!$entity) {
            throw $this->createNotFoundException('Associado Não Existe');
        }

        $pagamento = new Pagamento();
        $form = $this->createCreateForm($pagamento, $idCR);

        $dados = $this->pagamentoInfo($entity->getSalario());
        return $this->render('UsokoSIGBundle:Pagamento:Atendente/PA_NovoPag.html.twig', array(
                    'entity' => $entity,
                    'form' => $form->createView(),
                    'dados' => $dados,
        ));
    }

    /**
     * Finds and displays a Pagamento entity.
     *
     */
    public function showAction($id) {
        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('UsokoSIGBundle:Pagamento')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Pagamento entity.');
        }

        $deleteForm = $this->createDeleteForm($id);

        return $this->render('UsokoSIGBundle:Pagamento:show.html.twig', array(
                    'entity' => $entity,
                    'delete_form' => $deleteForm->createView(),
        ));
    }

    /**
     * Displays a form to edit an existing Pagamento entity.
     *
     */
    public function editAction($id) {

        $idN = \Usoko\SIGBundle\General\CryptGeneral::decrypt($this, $id);
        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('UsokoSIGBundle:Pagamento')->find($idN);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Pagamento entity.');
        }

        $editForm = $this->createEditForm($entity);
        $deleteForm = $this->createDeleteForm($id);

        return $this->render('UsokoSIGBundle:Pagamento:edit.html.twig', array(
                    'entity' => $entity,
                    'edit_form' => $editForm->createView(),
                    'delete_form' => $deleteForm->createView(),
        ));
    }

    /**
     * Creates a form to edit a Pagamento entity.
     *
     * @param Pagamento $entity The entity
     *
     * @return \Symfony\Component\Form\Form The form
     */
    private function createEditForm(Pagamento $entity) {
        $form = $this->createForm(new PagamentoType(), $entity, array(
            'action' => $this->generateUrl('pagamento_update', array('id' => $entity->getId())),
            'method' => 'PUT',
        ));

        $form->add('submit', 'submit', array('label' => 'Editar'));

        return $form;
    }

    /**
     * Edits an existing Pagamento entity.
     *
     */
    public function updateAction(Request $request, $id) {
        $em = $this->getDoctrine()->getManager();
        $em->getConnection()->beginTransaction();

        $entity = $em->getRepository('UsokoSIGBundle:Pagamento')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Pagamento entity.');
        }

        $deleteForm = $this->createDeleteForm($id);
        $editForm = $this->createEditForm($entity);
        $editForm->handleRequest($request);

        if ($editForm->isValid()) {

            $salario = $entity->getAssociado()->getSalario();
            $parte = $entity->getValor();
            $percentagem = MathGeneral::calcPercentagem($salario, $parte);

            $entity->setPercentagem($percentagem);

            $quota = $entity->getQuota()->getPercentagem();
            if ($percentagem < $quota) {
                $entity->setEstado('N'); // N/Liquidada
            } else {
                $entity->setEstado('L'); // Liquidada
            }

            $em->flush();
            AuditoriaGeneral::auditar($this, $em, $entity, "Pagamento", "U", "Actualização de Um Pagamento");
            $em->getConnection()->commit();
            return $this->redirect($this->generateUrl('pagamento'));
        }

        return $this->render('UsokoSIGBundle:Pagamento:edit.html.twig', array(
                    'entity' => $entity,
                    'edit_form' => $editForm->createView(),
                    'delete_form' => $deleteForm->createView(),
        ));
    }

    /**
     * Deletes a Pagamento entity.
     *
     */
    public function deleteAction(Request $request, $id) {
        $form = $this->createDeleteForm($id);
        $form->handleRequest($request);

        if ($form->isValid()) {
            $em = $this->getDoctrine()->getManager();
            $entity = $em->getRepository('UsokoSIGBundle:Pagamento')->find($id);

            if (!$entity) {
                throw $this->createNotFoundException('Unable to find Pagamento entity.');
            }

            AuditoriaGeneral::auditar($this, $em, $entity, "Pagamento", "D", "Remoção de Um Pagamento");

            $em->remove($entity);
            $em->flush();

            $em->getConnection()->commit();
        }

        return $this->redirect($this->generateUrl('pagamento'));
    }

    /**
     * Creates a form to delete a Pagamento entity by id.
     *
     * @param mixed $id The entity id
     *
     * @return \Symfony\Component\Form\Form The form
     */
    private function createDeleteForm($id) {
        return $this->createFormBuilder()
                        ->setAction($this->generateUrl('pagamento_delete', array('id' => $id)))
                        ->setMethod('DELETE')
                        ->add('submit', 'submit', array('label' => 'Eliminar'))
                        ->getForm()
        ;
    }

    public function myPagamentoAction($id) {
        $id = $this->get('nzo_url_encryptor')->decrypt($id);
        $em = $this->getDoctrine()->getManager();
        $entity = $em->getRepository('UsokoSIGBundle:Associado')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Associado entity.');
        }

        $entities = $em->getRepository('UsokoSIGBundle:Pagamento')->findBy(array('associado' => $id), array('mesAno' => 'DESC', 'tipopagamento' => 'DESC'));
        //$entities = $em->getRepository('UsokoSIGBundle:Pagamento')->findByAssociado($id,array('mesAno' => 'asc','tipopagamento' => 'asc' ));

        $dados = $this->pagamentoInfo($entity->getSalario());

        return $this->render('UsokoSIGBundle:Pagamento:PA_ListaPag.html.twig', array(
                    'entity' => $entity,
                    'entities' => $entities,
                    'dados' => $dados,
        ));
    }

    public function myPagamentoAtAction($id) {

        $id = $this->get('nzo_url_encryptor')->decrypt($id);
        $em = $this->getDoctrine()->getManager();
        $entity = $em->getRepository('UsokoSIGBundle:Associado')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Associado entity.');
        }

        $entities = $em->getRepository('UsokoSIGBundle:Pagamento')->findBy(array('associado' => $id), array('mesAno' => 'DESC', 'tipopagamento' => 'DESC'));

        $dados = $this->pagamentoInfo($entity->getSalario());

        return $this->render('UsokoSIGBundle:Pagamento:Atendente/PA_ListaPag.html.twig', array(
                    'entity' => $entity,
                    'entities' => $entities,
                    'dados' => $dados,
        ));
    }

    public function showMyAction($id, $pagamento) {

        $id = $this->get('nzo_url_encryptor')->decrypt($id);
        $pagamento = $this->get('nzo_url_encryptor')->decrypt($pagamento);

        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('UsokoSIGBundle:Associado')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Associado entity.');
        }

        $pagamento = $em->getRepository('UsokoSIGBundle:Pagamento')->find($pagamento);

        if (!$pagamento) {
            throw $this->createNotFoundException('Unable to find Pagamento entity.');
        }

        $dados = $this->pagamentoInfo($entity->getSalario());
        return $this->render('UsokoSIGBundle:Pagamento:PA_DadosPag.html.twig', array(
                    'entity' => $entity,
                    'pagamento' => $pagamento,
                    'dados' => $dados,
        ));
    }

    public function pagamentoInfo($salario) {

        $em = $this->getDoctrine()->getManager();

        $tipopagamento = new \Usoko\SIGBundle\Entity\Tipopagamento();
        $tipopagamento = $em->getRepository('UsokoSIGBundle:TipoPagamento')->findOneBy(array('tipo' => 'Q'));

        if (!$tipopagamento) {
            return false;
        }
        $valor = MathGeneral::calcParte($salario, $tipopagamento->getValor());
        $dados = array('valorLiquido' => $valor, 'percentagem' => $tipopagamento->getValor(), 'moeda' => $tipopagamento->getMoeda());
        return $dados;
    }

    public function associadoTObeneficiario($associado) {

        $em = $this->getDoctrine()->getManager();
        $pagamentos = $em->getRepository('UsokoSIGBundle:Pagamento')->findBy(array('estado' => 'L', 'associado' => $associado));
        return (count($pagamentos) >= VarsGeneral::CONS_ASSOCIADO2BENEFICIARIO);
    }

    public function hasJoia($associado, $em) {

        return $em->getRepository('UsokoSIGBundle:Pagamento')
                        ->createQueryBuilder('p')
                        ->select('p')
                        ->leftJoin('p.associado', 'a')->leftJoin('p.tipopagamento', 't')
                        ->where('a.id = :associado AND t.tipo = :tipo')
                        ->setParameter('associado', $associado)->setParameter('tipo', 'J')
                        ->getQuery()->getResult();
    }

    public function hasPaid($entity, $em) {

        $tipo = $entity->getTipopagamento()->getTipo();
        $associado = $entity->getAssociado()->getId();
        $mesAno = $entity->getMesAno();

        return $em->getRepository('UsokoSIGBundle:Pagamento')
                        ->createQueryBuilder('p')
                        ->select('p')
                        ->leftJoin('p.associado', 'a')->leftJoin('p.tipopagamento', 't')
                        ->where('a.id = :associado AND t.tipo = :tipo AND p.estado = :estado AND p.mesAno = :mesano')
                        ->setParameter('associado', $associado)->setParameter('tipo', $tipo)
                        ->setParameter('estado', 'L')->setParameter('mesano', $mesAno)
                        ->getQuery()->getResult();
    }

    public function redirectPayment($entity, $em) {

        $rs['return'] = false;
        $rs['code'] = 29;
        $rs['msg'] = "Valor Insuficiente";
        $rs['campo'] = 'valor';
        $tipopagamento = $entity->getTipopagamento();

        if ($tipopagamento->getTipo() == 'Q') {
            // ja tem joia
            if (!$this->hasJoia($entity->getAssociado()->getId(), $em)) {
                $rs['return'] = false;
                $rs['code'] = 28;
                $rs['msg'] = "Associado sem registos de pagamento da joia";
                $rs['campo'] = 'mesAno';
                return $rs;
            }
            // mes pago ?
            if ($this->hasPaid($entity, $em)) {
                $rs['return'] = false;
                $rs['code'] = 30;
                $rs['msg'] = "Quota já liquidada neste mês";
                $rs['campo'] = 'mesAno';
                return $rs;
            }
            // Quota 
            $salario = $entity->getAssociado()->getSalario();
            $percentagem = MathGeneral::calcPercentagem($salario, $entity->getValor());

            if ($percentagem >= $tipopagamento->getValor()) {
                $entity->setPercentagem($percentagem);
                $rs['return'] = true;
                $rs['entity'] = $entity;
                return $rs;
            }
            return $rs;
        } elseif ($tipopagamento->getTipo() == 'J') {

            if ($this->hasJoia($entity->getAssociado()->getId(), $em)) {
                $rs['return'] = false;
                $rs['code'] = 28;
                $rs['msg'] = "Associado já contém registos de pagamento da joia";
                $rs['campo'] = 'tipopagamento';
                return $rs;
            }
            // Joia 
            if ($entity->getValor() >= $tipopagamento->getValor()) {

                $rs['return'] = true;
                $rs['entity'] = $entity;
                return $rs;
            }
            return $rs;
        } elseif ($tipopagamento->getTipo() == 'C') {
            // PP 
            if ($this->hasPaid($entity, $em)) {
                $rs['return'] = false;
                $rs['code'] = 30;
                $rs['msg'] = "Pagamento já liquidado neste mês";
                $rs['campo'] = 'mesAno';
                return $rs;
            }
            if ($entity->getValor() >= $tipopagamento->getValor()) {
                $rs['return'] = true;
                $rs['entity'] = $entity;
                return $rs;
            }
        } else {
            // PP
            $rs['return'] = true;
            $rs['entity'] = $entity;
            return $rs;
        }
        return $rs;
    }

    private function valor($request) {
        $valor = $request->request->get('usoko_sigbundle_pagamento')['valor'];
        $valor = NumeroReal::converter($valor);
        return $valor;
    }

}
